#include "GridGenerator.h"
#include "Algo/Copy.h"
#include "BaseTile.h"
#include "Components/BillboardComponent.h"
#include "UObject/UnrealType.h"

/**
 * @brief Construct a new AGridGenerator::AGridGenerator object
 *
 */
AGridGenerator::AGridGenerator() {
  Rows = 25;
  Columns = 10;
  NumberOfTiles = 0;
}

/**
 * @brief when level begins, update number of tiles
 *
 */
void AGridGenerator::BeginPlay() {
  Super::BeginPlay();

  setNumberOfTiles();
}

/**
 * @brief sets the number of tiles in the grid
 *
 * @return int32
 */
int32 AGridGenerator::setNumberOfTiles() {
  NumberOfTiles = Rows * Columns;
  return NumberOfTiles;
}

/**
 * @brief get array of the tiles in the grid
 *
 * @return const TArray<ABaseTile *>&
 */
const TArray<ABaseTile *> &AGridGenerator::getTilesArray() const {
  return Tiles;
}

/**
 * @brief returns only the tiles that can be spawned on
 *
 * @return TArray<ABaseTile *>
 */
TArray<ABaseTile *> AGridGenerator::getSpawnableTiles() const {
  TArray<ABaseTile *> SpawnableTiles;

  // creates a copy of the array, but only inserts the tiles that have the bool
  // spawnable
  Algo::CopyIf(Tiles, SpawnableTiles,
               [](const ABaseTile *Tile) { return Tile && Tile->bSpawnable; });

  return SpawnableTiles;
}

/**
 * @brief checks if the location given has a tile that can be spawned on
 *
 * @param Location
 * @return true
 * @return false
 */
bool AGridGenerator::isTileSpawnableAtLocation(const FVector &Location) const {
  TArray<ABaseTile *> SpawnableTiles = getSpawnableTiles();

  // Check if the location matches any spawnable tile's location
  for (const ABaseTile *Tile : SpawnableTiles) {
    if (Tile && Tile->GetActorLocation().Equals(
                    Location, 90.0f)) { // might need to adjust this value
      return true;
    }
  }

  return false;
}

#if WITH_EDITOR
/**
 * @brief only relevant for blueprint
 *
 * @param PropertyChangedEvent
 */
void AGridGenerator::PostEditChangeProperty(
    FPropertyChangedEvent &PropertyChangedEvent) {
  Super::PostEditChangeProperty(PropertyChangedEvent);

  FName PropertyName = PropertyChangedEvent.Property
                           ? PropertyChangedEvent.Property->GetFName()
                           : NAME_None;
  if (PropertyName == GET_MEMBER_NAME_CHECKED(AGridGenerator, Rows) ||
      PropertyName == GET_MEMBER_NAME_CHECKED(AGridGenerator, Columns)) {
    setNumberOfTiles();
  }
}
#endif
