// Fill out your copyright notice in the Description page of Project Settings.

#include "CrossNChompGameMode.h"
#include "EngineUtils.h"
#include "PlayerCharacter.h"

#include "Algo/RandomShuffle.h"
#include "Apple.h"
#include "BaseTile.h"
#include "Coin.h"
#include "Ghost.h"
#include "GridGenerator.h"
#include "Item.h"
#include "Kismet/GameplayStatics.h"

#include <algorithm>

ACrossNChompGameMode::ACrossNChompGameMode() {

  // Set default class for Ghost (can be overridden in the editor)
  static ConstructorHelpers::FClassFinder<AGhost> GhostBlueBPClass(
      TEXT("/Game/Blueprints/BP_Ghost_Blue"));
  static ConstructorHelpers::FClassFinder<AGhost> GhostRedBPClass(
      TEXT("/Game/Blueprints/BP_Ghost_Red"));
  static ConstructorHelpers::FClassFinder<AGhost> GhostPinkBPClass(
      TEXT("/Game/Blueprints/BP_Ghost_Pink"));
  static ConstructorHelpers::FClassFinder<AGhost> GhostGreenBPClass(
      TEXT("/Game/Blueprints/BP_Ghost_Green"));

  if (GhostBlueBPClass.Succeeded()) {
    GhostClasses.Add(GhostBlueBPClass.Class);
  }
  if (GhostRedBPClass.Succeeded()) {
    GhostClasses.Add(GhostRedBPClass.Class);
  }
  if (GhostPinkBPClass.Succeeded()) {
    GhostClasses.Add(GhostPinkBPClass.Class);
  }
  if (GhostGreenBPClass.Succeeded()) {
    GhostClasses.Add(GhostGreenBPClass.Class);
  }

  // Set a default value for PlayerClass
  static ConstructorHelpers::FClassFinder<APlayerCharacter> DefaultPlayerClass(
      TEXT("/Game/Blueprints/BP_Player"));
  if (DefaultPlayerClass.Succeeded()) {
    PlayerClass = DefaultPlayerClass.Class;
  }

  // Use PlayerClass as the DefaultPawnClass
  DefaultPawnClass = PlayerClass;
}

void ACrossNChompGameMode::BeginPlay() {
  Super::BeginPlay();

  // Spawn the ghosts at the start of the game
  // SpawnGhosts();
  spawnItems();
}

void ACrossNChompGameMode::EndGame() {
  // Logic to end the game
}

// ensures the grid generator exists and is properly initialized
AGridGenerator *ACrossNChompGameMode::checkGrid() {
  for (TActorIterator<AGridGenerator> It(GetWorld()); It; ++It) {
    if (*It) {
      return *It;
    }
  }

  return nullptr;
}

/**
 * @brief function to spawn items on the grid
 *
 */
void ACrossNChompGameMode::spawnItems() {

  // ensure all classes are set
  if (GhostClasses.Num() == 0) {
    return;
  }
  if (!OrangeClass || !AppleClass || !StrawberryClass || !BananaClass ||
      !CoinClass) {

    return;
  }

  // ensure grid exists
  AGridGenerator *GridGenerator = checkGrid();
  if (!GridGenerator || GridGenerator->Tiles.Num() == 0)
    return;

  // by accessing a reference of the array and shuffling it we are saving
  // performance because we don't need to pick a random number for each spawn.
  // Additionally this ensures no duplicate spawn - we're accessing a copy of
  // the tiles array that only has tiles in it that are spawnable

  TArray<ABaseTile *> TilesArray = GridGenerator->getSpawnableTiles();
  Algo::RandomShuffle(TilesArray);

  // could have also been accessed via TilesArray.Num()-1;
  int SpawnAmount = TilesArray.Num() - 1;

  int16 GhostSpawning = 0;

  // Retrieve the SpawnFactor from the Apple class (without instantiating it)
  AApple *AppleInstance =
      AppleClass
          ->GetDefaultObject<AApple>(); // Get the default object of the class
  int32 MaxAppleSpawn = AppleInstance ? AppleInstance->getSpawnFactor() : 0;

  ABanana *BananaInstance = BananaClass->GetDefaultObject<ABanana>();
  int32 MaxBananaSpawn = BananaInstance ? BananaInstance->getSpawnFactor() : 0;

  AOrange *OrangeInstance = OrangeClass->GetDefaultObject<AOrange>();
  int32 MaxOrangeSpawn = OrangeInstance ? OrangeInstance->getSpawnFactor() : 0;

  AStrawberry *StrawberryInstance =
      StrawberryClass->GetDefaultObject<AStrawberry>();
  int32 MaxStrawberrySpawn =
      StrawberryInstance ? StrawberryInstance->getSpawnFactor() : 0;

  int16 AppleSpawning = 0;
  int16 BananaSpawning = 0;
  int16 OrangeSpawning = 0;
  int16 StrawberrySpawning = 0;

  // loop i
  int32 i = 0;
  // index to access from randomised TArray
  int32 TileIndex = 0;

  // looping through number of items to be set
  // TODO: include ghosts in the spawn - they should also be the first to spawn
  while (i < SpawnAmount) {
    ABaseTile *RandomTile = TilesArray[TileIndex];

    if (RandomTile) {
      FVector SpawnLocation = RandomTile->GetActorLocation();
      // let's randomise this a bit later
      FRotator SpawnRotation = FRotator::ZeroRotator;
      FActorSpawnParameters SpawnParams;

      // first spawn all the ghosts until the spawnFactor is reached
      if (GhostSpawning <= NumberOfGhosts) {
        TSubclassOf<AGhost> RandomGhostClass =
            GhostClasses[FMath::RandRange(0, GhostClasses.Num() - 1)];
        GetWorld()->SpawnActor<AGhost>(RandomGhostClass, SpawnLocation,
                                       SpawnRotation, SpawnParams);
        ++GhostSpawning;
      } else if (AppleSpawning <= MaxAppleSpawn) {
        //  spawn all the fruits until the spawnFactor is reached

        AApple *SpawnedApple = GetWorld()->SpawnActor<AApple>(
            AppleClass, SpawnLocation, SpawnRotation, SpawnParams);

        ++AppleSpawning;
      } else if (BananaSpawning <= MaxBananaSpawn) {

        ABanana *SpawnedBanana = GetWorld()->SpawnActor<ABanana>(
            BananaClass, SpawnLocation, SpawnRotation, SpawnParams);

        ++BananaSpawning;
      } else if (OrangeSpawning <= MaxOrangeSpawn) {

        AOrange *SpawnedOrange = GetWorld()->SpawnActor<AOrange>(
            OrangeClass, SpawnLocation, SpawnRotation, SpawnParams);

        ++OrangeSpawning;
      } else if (StrawberrySpawning <= MaxStrawberrySpawn) {

        AStrawberry *SpawnedStrawberry = GetWorld()->SpawnActor<AStrawberry>(
            StrawberryClass, SpawnLocation, SpawnRotation, SpawnParams);

        ++StrawberrySpawning;
      }

      else {
        // then for the rest of the tiles, spawn coins

        ACoin *SpawnedCoin = GetWorld()->SpawnActor<ACoin>(
            CoinClass, SpawnLocation, SpawnRotation, SpawnParams);
      }
    }

    ++i;
    ++TileIndex;
  }
}