#include "PlayerCharacter.h"
#include "BaseTile.h"
#include "Components/CapsuleComponent.h"
#include "CrossNChompGameMode.h"
#include "EnhancedInputComponent.h"
#include "EnhancedInputSubsystems.h"
#include "GameFramework/CharacterMovementComponent.h"
#include "Ghost.h"
#include "GridGenerator.h"
#include "InputAction.h"
#include "InputMappingContext.h"
#include "Item.h"
#include "Kismet/GameplayStatics.h"
#include "TimerManager.h"

/**
 * @brief Construct a new APlayerCharacter::APlayerCharacter object
 *
 */
APlayerCharacter::APlayerCharacter() {
  // Set default values
  Lives = 3;
  CurrentLives = Lives;
  CurrentScore = 0;

  GetCharacterMovement()->MaxWalkSpeed = 100.0f; // 100 equals a tile
  GetCharacterMovement()->bOrientRotationToMovement = true;
  GridGenerator = nullptr; // Initialize to null
}

/**
 * @brief ensures character snaps to grid
 *
 * @param DeltaTime
 */
void APlayerCharacter::Tick(float DeltaTime) {
  Super::Tick(DeltaTime);

  FVector Location = GetActorLocation();
  Location.X = FMath::RoundToInt(Location.X / MovementSize) * MovementSize;
  Location.Y = FMath::RoundToInt(Location.Y / MovementSize) * MovementSize;
  SetActorLocation(Location);
}

/**
 * @brief checks for collision and triggers corresponding function
 *
 * @param OtherActor
 */
void APlayerCharacter::NotifyActorBeginOverlap(AActor *OtherActor) {
  Super::NotifyActorBeginOverlap(OtherActor);

  if (OtherActor) {
    AGhost *Ghost = Cast<AGhost>(OtherActor);
    AItem *Item = Cast<AItem>(OtherActor);

    // if collides with ghost, subtract a live and reset player position
    if (Ghost) {
      updateLives(1);
      playerDies();
    }

    // if collides with item, trigger eat item function
    else if (Item) {
      eatItem(Item);
    }
  }
}

/**
 * @brief when collision with item: updates the score and then destroys it
 *
 * @param Item
 */
void APlayerCharacter::eatItem(AItem *Item) {
  // checks for the value of item eaten
  updateCurrentScore(Item->getValue());

  // then destroys it
  Item->Destroy();
}

/**
 * @brief updates the lives left
 *
 * @param ToSubtract
 */
void APlayerCharacter::updateLives(int ToSubtract) {
  CurrentLives = CurrentLives - ToSubtract;
}

/**
 * @brief updates the score
 *
 * @param ScoreToAdd
 */
void APlayerCharacter::updateCurrentScore(int ScoreToAdd) {
  CurrentScore = CurrentScore + ScoreToAdd;
}

/**
 * @brief returns the amount of lives left
 *
 * @return int
 */
int APlayerCharacter::getCurrentLives() const { return CurrentLives; }

/**
 * @brief returns the current score
 *
 * @return int
 */
int APlayerCharacter::getCurrentScore() const { return CurrentScore; }

/**
 * @brief on level start, set the controller
 *
 */
void APlayerCharacter::BeginPlay() {
  Super::BeginPlay();
  locateGridGenerator();

  // Add Input Mapping Context
  if (APlayerController *PlayerController =
          Cast<APlayerController>(GetController())) {
    if (UEnhancedInputLocalPlayerSubsystem *Subsystem =
            ULocalPlayer::GetSubsystem<UEnhancedInputLocalPlayerSubsystem>(
                PlayerController->GetLocalPlayer())) {
      Subsystem->AddMappingContext(InputMappingContext, 0);
    }
  }
}

/**
 * @brief sets input controls
 *
 * @param PlayerInputComponent
 */
void APlayerCharacter::SetupPlayerInputComponent(
    UInputComponent *PlayerInputComponent) {
  Super::SetupPlayerInputComponent(PlayerInputComponent);

  if (UEnhancedInputComponent *EnhancedInputComponent =
          Cast<UEnhancedInputComponent>(PlayerInputComponent)) {

    EnhancedInputComponent->BindAction(IA_Move, ETriggerEvent::Triggered, this,
                                       &APlayerCharacter::handleMoveInput);
  }

  GetWorld()->GetTimerManager().SetTimer(InactivityTimerHandle, this,
                                         &APlayerCharacter::checkInactivity,
                                         10.0f, true);
}

/**
 * @brief on input get the current location and set the next location according
 * to the tile
 *
 * @param Value
 */
void APlayerCharacter::handleMoveInput(const FInputActionValue &Value) {
  bHasInput = true;
  FVector currentLocation = GetActorLocation();

  // to prevent multiple inputs
  if (bIsMoving)
    return;

  // get input vector
  FVector2D input = Value.Get<FVector2D>();
  if (input.IsZero())
    return;

  // sets moving flag to true so no double input
  bIsMoving = true;

  FVector direction = FVector(input.X, input.Y, 0.0f).GetSafeNormal();
  FVector targetLocation = GetActorLocation() + direction * MovementSize;

  // Move to the target location if the tile can be moved onto
  bool bIsSpawnable = GridGenerator->isTileSpawnableAtLocation(targetLocation);
  if (bIsSpawnable) {
    GetWorld()->GetTimerManager().SetTimer(
        InactivityTimerHandle,
        [this, targetLocation]() {
          SetActorLocation(targetLocation);
          bIsMoving = false;
        },
        MovementDuration, false);

  } else {
    // stay at current location if wall
    GetWorld()->GetTimerManager().SetTimer(
        InactivityTimerHandle,
        [this, currentLocation]() {
          SetActorLocation(currentLocation);
          bIsMoving = false;
        },
        MovementDuration, false);
  }
}

/**
 * @brief triggers game end if no lives are left and destroys player - otherwise
 * reset position
 *
 */
void APlayerCharacter::playerDies() {
  // Check if the player has no lives left
  if (CurrentLives <= 0) {
    // End the game
    ACrossNChompGameMode *GameMode =
        Cast<ACrossNChompGameMode>(UGameplayStatics::GetGameMode(this));
    if (GameMode) {
      GameMode->EndGame();
    }

    // Destroy the player
    Destroy();
  } else {
    // Reset player position - using 0/0/0 because it's the first tile in the
    // grid and the initial start value
    FVector respawnLocation = FVector(0.0f, 0.0f, 0.0f);
    this->SetActorLocation(respawnLocation);

    // resets movement flag
    bIsMoving = false;
  }
}

/**
 * @brief triggers player death if no input is done in 10 seconds
 *
 */
void APlayerCharacter::checkInactivity() {
  if (!bHasInput) {
    updateLives(3); // Trigger player death if no input
    playerDies();
  } else {
    bHasInput = false; // resets the input flag for the next check
  }
}

void APlayerCharacter::BeginDestroy() {
  Super::BeginDestroy();

  // clears the timer
  if (GetWorld()) {
    GetWorld()->GetTimerManager().ClearTimer(InactivityTimerHandle);
  }
}

/**
 * @brief reference to grid
 *
 * @return AGridGenerator*
 */
AGridGenerator *APlayerCharacter::locateGridGenerator() {
  // Find the grid generator in the world
  for (TActorIterator<AGridGenerator> It(GetWorld()); It; ++It) {
    if (*It) {
      GridGenerator = *It;
      return GridGenerator;
    }
  }

  return nullptr;
}
