// Fill out your copyright notice in the Description page of Project Settings.

#include "Ghost.h"
#include "BaseTile.h"
#include "Components/CapsuleComponent.h"
#include "Components/SphereComponent.h"
#include "EngineUtils.h"
#include "GridGenerator.h"

AGhost::AGhost() {
  // Create and set up the collision component

  // collision mesh
  CollisionMesh = CreateDefaultSubobject<UBoxComponent>(TEXT("Collision Box"));
  CollisionMesh->InitBoxExtent(FVector(50));
  CollisionMesh->SetCollisionProfileName(TEXT("Trigger Collision Box"));

  CollisionMesh->SetCollisionObjectType(ECC_Pawn);
  CollisionMesh->SetCollisionResponseToChannel(ECC_Pawn, ECR_Overlap);
  CollisionMesh->SetGenerateOverlapEvents(true);

  BaseMesh = CreateDefaultSubobject<UStaticMeshComponent>(TEXT("Base Mesh"));
  BaseMesh->SetGenerateOverlapEvents(true);
  BaseMesh->SetCollisionObjectType(ECC_Pawn);

  BaseMesh->SetupAttachment(CollisionMesh);
  RootComponent = CollisionMesh;
  // static mesh

  // default mesh, can be overwritten
  static ConstructorHelpers::FObjectFinder<UStaticMesh> MeshAsset(
      TEXT("/Engine/BasicShapes/Cube.Cube"));
  if (MeshAsset.Succeeded()) {
    BaseMesh->SetStaticMesh(MeshAsset.Object);
  }

  GridGenerator = nullptr; // Initialize to null
}

void AGhost::Tick(float DeltaTime) {
  Super::Tick(DeltaTime);

  // Snap position to the grid
  FVector Location = GetActorLocation();
  Location.X = FMath::RoundToInt(Location.X / MovementSize) * MovementSize;
  Location.Y = FMath::RoundToInt(Location.Y / MovementSize) * MovementSize;
  SetActorLocation(Location);
}

/**
 * @brief moves randomly on x lane
 *
 */
void AGhost::moveRandomly() {
  FVector currentLocation = GetActorLocation();
  FVector direction = getRandomdirection();
  FVector targetLocation = currentLocation + direction * MovementSize;

  // Move to the target location
  // SetActorLocation(TargetLocation);

  // if the tile can be moved to, just set the position
  if (canMoveTo(targetLocation)) {
    SetActorLocation(targetLocation);
  } else {
    // recursion kills unreal, so we're simply reversing the direction by
    // multiplying the vector with -1

    direction = direction * -1;

    targetLocation = currentLocation + direction * MovementSize;
    if (canMoveTo(targetLocation)) {
      SetActorLocation(targetLocation);
    }
  }
  LastDirection = direction; // Store the last direction
}

void AGhost::BeginPlay() {
  Super::BeginPlay();
  locateGridGenerator();
  // every x (MovementInterval) seconds, move randomly
  GetWorld()->GetTimerManager().SetTimer(
      MovementTimerHandle, this, &AGhost::moveRandomly, MovementInterval, true);
}

/**
 * @brief checks if tile can be moved to
 *
 * @param TargetLocation
 * @return true
 * @return false
 */
bool AGhost::canMoveTo(FVector TargetLocation) {

  if (!GridGenerator) {
    return false; // Cannot move without a valid grid generator
  }
  bool bIsSpawnable = GridGenerator->isTileSpawnableAtLocation(TargetLocation);
  // Use the GridGenerator's method to check spawnability

  return bIsSpawnable;
}

/**
 * @brief random direction to pick from
 *
 * @return FVector
 */
FVector AGhost::getRandomdirection() const {
  TArray<FVector> directions = {
      FVector(0, 1, 0), // Left
      FVector(0, -1, 0) // Right

  };

  int32 RandomIndex = FMath::RandRange(0, directions.Num() - 1);
  return directions[RandomIndex];
}

/**
 * @brief pointer reference to grid
 *
 * @return AGridGenerator*
 */
AGridGenerator *AGhost::locateGridGenerator() {
  // Find the grid generator in the world
  for (TActorIterator<AGridGenerator> It(GetWorld()); It; ++It) {
    if (*It) {
      GridGenerator = *It;
      return GridGenerator;
    }
  }

  return nullptr;
}
